/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2017 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BoostGraphCreatorUnitTest.cpp
 *
 *  Created on: May 19, 2017
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#include "Atom/atom.hpp"
#include "Graph/BoostGraphCreator.hpp"
#include "Graph/BreadthFirstSearchGatherer.hpp"
#include "molecule.hpp"
#include "Element/periodentafel.hpp"
#include "World.hpp"

#include "BoostGraphCreatorUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( BoostGraphCreatorTest );


void BoostGraphCreatorTest::setUp()
{
  BGCreator = new BoostGraphCreator;

  // construct element
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  carbon = World::getInstance().getPeriode()->FindElement(6);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");
  CPPUNIT_ASSERT(carbon != NULL && "could not find element carbon");

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL && "could not create molecule");
  atom *Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(5., 5., 5. ));
  TestMolecule->AddAtom(Walker);

  atom *OtherWalker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(OtherWalker != NULL && "could not create atom");
  OtherWalker->setType(carbon);
  Walker->setPosition(Vector(6.5, 5., 5. ));
  TestMolecule->AddAtom(OtherWalker);
  Walker->addBond(OtherWalker);

  atom *HWalker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(HWalker != NULL && "could not create atom");
  HWalker->setType(hydrogen);
  HWalker->setPosition(Vector(4.3, 4.5, 5. ));
  TestMolecule->AddAtom(HWalker);
  Walker->addBond(HWalker);

  HWalker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(HWalker != NULL && "could not create atom");
  HWalker->setType(hydrogen);
  HWalker->setPosition(Vector(4.3, 5.5, 5. ));
  TestMolecule->AddAtom(HWalker);
  Walker->addBond(HWalker);

  HWalker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(HWalker != NULL && "could not create atom");
  HWalker->setType(hydrogen);
  HWalker->setPosition(Vector(7.2, 4.5, 5. ));
  TestMolecule->AddAtom(HWalker);
  OtherWalker->addBond(HWalker);

  HWalker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(HWalker != NULL && "could not create atom");
  HWalker->setType(hydrogen);
  HWalker->setPosition(Vector(7.2, 5.5, 5. ));
  TestMolecule->AddAtom(HWalker);
  OtherWalker->addBond(HWalker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 6 );

};


void BoostGraphCreatorTest::tearDown()
{
  delete BGCreator;

  // remove molecule
  World::getInstance().destroyMolecule(TestMolecule);
  // note that all the atoms, molecules, the tafel and the elements
  // are all cleaned when the world is destroyed
  World::purgeInstance();
  logger::purgeInstance();
};

/** Tests whether setup works.
 */
void BoostGraphCreatorTest::SetupTest()
{
//  CPPUNIT_ASSERT_EQUAL (false, TestMolecule->empty());
}

/** Tests whether createFromRange() works.
 */
void BoostGraphCreatorTest::createFromRangeTest()
{
//  CPPUNIT_ASSERT_EQUAL (false, TestMolecule->empty());
};

/** Tests whether createFromMolecule() works.
 */
void BoostGraphCreatorTest::createFromMoleculeTest()
{
  BGCreator->createFromMolecule(*TestMolecule, BreadthFirstSearchGatherer::AlwaysTruePredicate);

  CPPUNIT_ASSERT_EQUAL ((size_t)6, BGCreator->getNumVertices());
  CPPUNIT_ASSERT_EQUAL ((size_t)5, BGCreator->getNumEdges());
};

/** Tests whether createFromAtoms() works.
 */
void BoostGraphCreatorTest::createFromAtomsTest()
{
  std::vector<const atom *> atoms;
  std::copy(TestMolecule->begin(), TestMolecule->end(), std::back_inserter(atoms));
  BGCreator->createFromAtoms(atoms, BreadthFirstSearchGatherer::AlwaysTruePredicate);

  CPPUNIT_ASSERT_EQUAL ((size_t)6, BGCreator->getNumVertices());
  CPPUNIT_ASSERT_EQUAL ((size_t)5, BGCreator->getNumEdges());
};

/** Tests whether adding and removing edges works.
 */
void BoostGraphCreatorTest::addremoveEdgeTest()
{
  typedef std::pair<int,int> E;

  E edges[] = { E(0,1), E(1,2), E(2,3), E(3,4) };
  const size_t no_nodes = 5;
  BGCreator->graph =
      BoostGraphCreator::UndirectedGraph(edges, edges + sizeof(edges) / sizeof(E), no_nodes);
  BGCreator->atomids_nodeids +=
      make_pair(0,0), make_pair(1,1), make_pair(2,2), make_pair(3,3), make_pair(4,4);
  for (size_t i=0;i<no_nodes;++i)
    boost::put(boost::get(boost::vertex_name, BGCreator->graph), boost::vertex(i, BGCreator->graph), i);

  CPPUNIT_ASSERT_EQUAL ((size_t)5, BGCreator->getNumVertices());
  CPPUNIT_ASSERT_EQUAL ((size_t)4, BGCreator->getNumEdges());

  bool status;
  // remove a valid edge
  {
    status = BGCreator->removeEdge((atomId_t)0,(atomId_t)1);
    CPPUNIT_ASSERT_EQUAL (true, status);
    status = BGCreator->addEdge((atomId_t)0,(atomId_t)1);
    CPPUNIT_ASSERT_EQUAL (true, status);
    // check again whether edge has really been added again
    status = BGCreator->removeEdge((atomId_t)0,(atomId_t)1);
    CPPUNIT_ASSERT_EQUAL (true, status);
    status = BGCreator->addEdge((atomId_t)0,(atomId_t)1);
    CPPUNIT_ASSERT_EQUAL (true, status);
  }

  // remove an invalid edge
  {
    status = BGCreator->removeEdge((atomId_t)0,(atomId_t)2);
    CPPUNIT_ASSERT_EQUAL (false, status);
  }

  // add a present edge
  {
    status = BGCreator->addEdge((atomId_t)0,(atomId_t)1);
    CPPUNIT_ASSERT_EQUAL (false, status);
  }
};
