/*
 * GaussianThermostat.cpp
 *
 *  Created on: Aug 18, 2010
 *      Author: crueger
 */

#include "GaussianThermostat.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "AtomSet.hpp"
#include "element.hpp"
#include "config.hpp"
#include <set>

GaussianThermostat::GaussianThermostat() : E(0),G(0)
{}

GaussianThermostat::~GaussianThermostat()
{}

double GaussianThermostat::scaleAtoms(config &configuration,unsigned int step,ATOMSET(std::list) atoms){
  return doScaleAtoms(configuration,step,atoms.begin(),atoms.end());
}

double GaussianThermostat::scaleAtoms(config &configuration,unsigned int step,ATOMSET(std::vector) atoms){
  return doScaleAtoms(configuration,step,atoms.begin(),atoms.end());
}

double GaussianThermostat::scaleAtoms(config &configuration,unsigned int step,ATOMSET(std::set) atoms){
  return doScaleAtoms(configuration,step,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double GaussianThermostat::doScaleAtoms(config &configuration,unsigned int step,ForwardIterator begin, ForwardIterator end){
  DoLog(2) && (Log() << Verbose(2) <<  "Applying Gaussian thermostat..." << endl);
  init(step,begin,end);
  double G_over_E = G/E;
  DoLog(1) && (Log() << Verbose(1) << "Gaussian Least Constraint constant is " << G_over_E << "." << endl);
  double ekin =0;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector &U = (*iter)->Trajectory.U.at(step);
    if ((*iter)->FixedIon == 0) {// even FixedIon moves, only not by other's forces
      U += configuration.Deltat * G_over_E * U;
      ekin += (*iter)->getType()->mass * U.NormSquared();
    }
  }
  return ekin;
}

template <class ForwardIterator>
void GaussianThermostat::init(unsigned int step,ForwardIterator begin, ForwardIterator end){
  E=0;
  G=0;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector &U = (*iter)->Trajectory.U.at(step);
    Vector &F = (*iter)->Trajectory.F.at(step);
    if ((*iter)->FixedIon == 0){ // even FixedIon moves, only not by other's forces
      G += U.ScalarProduct(F);
      E += U.NormSquared()*(*iter)->getType()->mass;
    }
  }
}

double GaussianThermostat::getE() const{
  return E;
}

double GaussianThermostat::getG() const{
  return G;
}
