/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FragmentQueueUnitTest.cpp
 *
 *  Created on: Oct 23, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "FragmentQueue.hpp"

#include "FragmentQueueUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( FragmentQueueTest );


void FragmentQueueTest::setUp()
{
  // Throw assertions
  ASSERT_DO(Assert::Throw);

  queue = new FragmentQueue();
}


void FragmentQueueTest::tearDown()
{
  delete queue;
}

/** UnitTest for working JobQueue
 */
void FragmentQueueTest::JobsTest()
{
  FragmentJob testJob;
  /// check for illegal id
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushJob(testJob), Assert::AssertionFailure );
#endif
  // set to valid id
  testJob.setId(1);
  testJob.outputfile = std::string("do something");

  CPPUNIT_ASSERT_EQUAL(false, queue->isJobPresent() );

#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
#else
  queue->pushJob(testJob);
#endif

  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL(true, queue->isJobPresent() );
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());
  FragmentQueue::ResultMap::const_iterator iter = queue->results.find((JobId_t)1);
  CPPUNIT_ASSERT( iter != queue->results.end() );
  CPPUNIT_ASSERT( FragmentQueue::NoResult == iter->second );

  // push same id again
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushJob(testJob), Assert::AssertionFailure );
#endif

  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());

  FragmentJob poppedJob;
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( poppedJob = queue->popJob() );
#else
  poppedJob = queue->popJob();
#endif
  CPPUNIT_ASSERT( poppedJob == testJob );

  CPPUNIT_ASSERT_EQUAL((size_t)0, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL(false, queue->isJobPresent() );
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());
  iter = queue->results.find((JobId_t)1);
  CPPUNIT_ASSERT( iter != queue->results.end() );
  CPPUNIT_ASSERT( FragmentQueue::NoResultQueued == iter->second );

#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->popJob(), Assert::AssertionFailure );
#endif
}

/** UnitTest for working ResultMap
 */
void FragmentQueueTest::ResultsTest()
{
  // prepare a job
  FragmentJob testJob;
  testJob.setId(1);
  testJob.outputfile = std::string("do something");
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
#else
  queue->pushJob(testJob);
#endif
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->popJob() );
#else
  queue->popJob();
#endif


  // prepare a result
  FragmentResult testResult(1);
  FragmentResult wrongIdResult(2);

  // check presence
  CPPUNIT_ASSERT( !queue->isResultPresent(1) );
  CPPUNIT_ASSERT( !queue->isResultPresent(2) );
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->getResult(1), Assert::AssertionFailure );
#endif

  /// check for wrong id
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushResult(wrongIdResult), Assert::AssertionFailure );
#endif
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushResult(testResult) );
#else
  queue->pushResult(testResult);
#endif

  // check presence again
  CPPUNIT_ASSERT( queue->isResultPresent(1) );

#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->getResult(1) );
#else
  queue->getResult(1);
#endif

  // check presence one more time
  CPPUNIT_ASSERT( !queue->isResultPresent(1) );
  CPPUNIT_ASSERT( !queue->isResultPresent(2) );
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->getResult(1), Assert::AssertionFailure );
#endif
}
