/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomDescriptor.cpp
 *
 *  Created on: Feb 5, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomDescriptor_impl.hpp"

#include "World.hpp"
#include "Atom/atom.hpp"
#include "CodePatterns/Observer/ObservedContainer_impl.hpp"

#include <boost/bind.hpp>

#include <iostream>

using namespace std;

typedef World::AtomSet::internal_iterator atoms_iter_t;
typedef World::AtomSet::const_iterator atoms_const_iter_t;

/************************ Forwarding object **************************************/


AtomDescriptor::AtomDescriptor(impl_ptr _impl) :
    impl(_impl)
{}

AtomDescriptor::AtomDescriptor(const AtomDescriptor& src) :
    impl(src.get_impl())
{}

AtomDescriptor::~AtomDescriptor()
{}

AtomDescriptor& AtomDescriptor::operator=(AtomDescriptor &src){
  if(&src!=this) {
    impl=src.get_impl();
  }
  return *this;
}

atom* AtomDescriptor::find(){
  return impl->find();
}

const atom* AtomDescriptor::find() const {
  return const_cast<const impl_t &>(*impl).find();
}

std::vector<atom*> AtomDescriptor::findAll(){
  return impl->findAll();
}

std::vector<const atom*> AtomDescriptor::findAll() const {
  return const_cast<const impl_t &>(*impl).findAll();
}

AtomDescriptor::impl_ptr AtomDescriptor::get_impl() const{
  return impl;
}




/**************************** implementation ********************/

AtomDescriptor_impl::AtomDescriptor_impl()
{
}

AtomDescriptor_impl::~AtomDescriptor_impl()
{
}

World::AtomSet& AtomDescriptor_impl::getAtoms(){
  return World::getInstance().atoms;
}

const World::AtomSet& AtomDescriptor_impl::getAtoms() const {
  return const_cast<const World &>(World::getInstance()).atoms;
}

atom* AtomDescriptor_impl::find() {
  World::AtomSet &atoms = getAtoms();
  atoms_iter_t res = find_if(atoms.begin_internal(),atoms.end_internal(),boost::bind(&AtomDescriptor_impl::predicate,this,_1));
  return (res!=atoms.end_internal())?((*res).second):0;
}

const atom* AtomDescriptor_impl::find() const {
  const World::AtomSet &atoms = getAtoms();
  atoms_const_iter_t res = find_if(atoms.begin(),atoms.end(),boost::bind(&AtomDescriptor_impl::predicate,this,_1));
  return (res!=atoms.end())?((*res).second):0;
}

vector<atom*> AtomDescriptor_impl::findAll() {
  vector<atom*> res;
  World::AtomSet atoms = getAtoms();
  for_each(atoms.begin_internal(),
           atoms.end_internal(),
           boost::bind(static_cast<void (AtomDescriptor_impl::*)(
               std::vector<atom*> *,
               std::pair<atomId_t,atom*>)>(&AtomDescriptor_impl::checkAndAdd),
                       this,&res,_1));
  return res;
}

vector<const atom*> AtomDescriptor_impl::findAll() const {
  vector<const atom*> res;
  const World::AtomSet &atoms = getAtoms();
  for_each(atoms.begin(),
           atoms.end(),
           boost::bind(static_cast<void (AtomDescriptor_impl::*)(
               std::vector<const atom*> *,
               std::pair<atomId_t,const atom*>) const>(&AtomDescriptor_impl::checkAndAdd),
                       this,&res,_1));
  return res;
}

void AtomDescriptor_impl::checkAndAdd(std::vector<atom*> *v,std::pair<atomId_t,atom*> p){
  if(predicate(p)){
    v->push_back(p.second);
  }
}

void AtomDescriptor_impl::checkAndAdd(std::vector<const atom*> *v,std::pair<atomId_t,const atom*> p) const{
  if(predicate(p)){
    v->push_back(p.second);
  }
}

/************************** Universe and Emptyset *****************/

AtomAllDescriptor_impl::AtomAllDescriptor_impl()
{}

AtomAllDescriptor_impl::~AtomAllDescriptor_impl()
{}

bool AtomAllDescriptor_impl::predicate(std::pair<atomId_t,const atom*>) const{
  return true;
}

AtomDescriptor AllAtoms(){
  return AtomDescriptor(AtomDescriptor::impl_ptr(new AtomAllDescriptor_impl));
}

AtomNoneDescriptor_impl::AtomNoneDescriptor_impl()
{}

AtomNoneDescriptor_impl::~AtomNoneDescriptor_impl()
{}

bool AtomNoneDescriptor_impl::predicate(std::pair<atomId_t,const atom*>) const{
  return false;
}

AtomDescriptor NoAtoms(){
  return AtomDescriptor(AtomDescriptor::impl_ptr(new AtomNoneDescriptor_impl));
}

/************************** Operator stuff ************************/

// AND
AtomAndDescriptor_impl::AtomAndDescriptor_impl(AtomDescriptor::impl_ptr _lhs, AtomDescriptor::impl_ptr _rhs) :
    lhs(_lhs), rhs(_rhs)
{}

AtomAndDescriptor_impl::~AtomAndDescriptor_impl()
{}

bool AtomAndDescriptor_impl::predicate(std::pair<atomId_t,const atom*> atom) const{
  return lhs->predicate(atom) && rhs->predicate(atom);
}
AtomDescriptor operator&&(const AtomDescriptor &lhs, const AtomDescriptor &rhs){
  AtomDescriptor::impl_ptr newImpl = AtomDescriptor::impl_ptr(new AtomAndDescriptor_impl(lhs.get_impl(),rhs.get_impl()));
  return AtomDescriptor(newImpl);
}

// OR
AtomOrDescriptor_impl::AtomOrDescriptor_impl(AtomDescriptor::impl_ptr _lhs ,AtomDescriptor::impl_ptr _rhs) :
    lhs(_lhs), rhs(_rhs)
{}

AtomOrDescriptor_impl::~AtomOrDescriptor_impl(){
}

bool AtomOrDescriptor_impl::predicate(std::pair<atomId_t,const atom*> atom) const{
  return lhs->predicate(atom) || rhs->predicate(atom);
}

AtomDescriptor  operator||(const AtomDescriptor &lhs, const AtomDescriptor &rhs){
  AtomDescriptor::impl_ptr newImpl = AtomDescriptor::impl_ptr(new AtomOrDescriptor_impl(lhs.get_impl(),rhs.get_impl()));
  return AtomDescriptor(newImpl);
}

// NOT

AtomNotDescriptor_impl::AtomNotDescriptor_impl(AtomDescriptor::impl_ptr _arg) :
    arg(_arg)
{}


AtomNotDescriptor_impl::~AtomNotDescriptor_impl()
{
}

bool AtomNotDescriptor_impl::predicate(std::pair<atomId_t,const atom*> atom) const{
 return !(arg->predicate(atom));
}

AtomDescriptor operator!(const AtomDescriptor &arg){
  AtomDescriptor::impl_ptr newImpl = AtomDescriptor::impl_ptr(new AtomNotDescriptor_impl(arg.get_impl()));
  return AtomDescriptor(newImpl);
}
