/*
 * Singleton.hpp
 *
 *  Created on: Mar 10, 2010
 *      Author: crueger
 */

#ifndef SINGLETON_HPP_
#define SINGLETON_HPP_

#include <cassert>
#include <boost/thread.hpp>

#include "defs.hpp"

/**
 * This template produces the generic singleton pattern using the CRTP idiom.
 */
template <class T, bool _may_create=true>
class Singleton
{
private:
  // simple auto_ptr that allows destruction of the object
  // std::auto_ptr cannot do this because the destructor of T is ussually private
  class ptr_t {
  public:
    ptr_t();
    ptr_t(T* _content);
    ~ptr_t();
    T& operator*();
    T* get();
    void reset(T* _content);
    void reset();
    ptr_t& operator=(const ptr_t& rhs);
  private:
    mutable T* content;
  };

  /**
   * this creator checks what it may or may not do
   */
  template<class creator_T, bool creator_may_create>
  struct creator_t {
    static creator_T* make();
    static void set(creator_T*&,creator_T*);
  };

  // specialization to allow fast creations

  template<class creator_T>
  struct creator_t<creator_T,true>{
    static creator_T* make(){
      return new creator_T();
    }

    static void set(creator_T*&,creator_T*){
      assert(0 && "Cannot set the Instance for a singleton of this type");
    }
  };

  template<class creator_T>
  struct creator_t<creator_T,false>{
    static creator_T* make(){
      assert(0 && "Cannot create a singleton of this type directly");
    }
    static void set(ptr_t& dest,creator_T* src){
      dest.reset(src);
    }
  };

public:

  // make the state of this singleton accessible
  static const bool may_create=_may_create;

  // this is used for creation
  typedef creator_t<T,_may_create> creator;

  static T& getInstance();
  static T* getPointer();

  static void purgeInstance();
  static T&   resetInstance();

  static void setInstance(T*);
protected:
  // constructor accessible by subclasses
  Singleton();

private:
  // private copy constructor to avoid unintended copying
  Singleton(const Singleton&);

  static boost::recursive_mutex instanceLock;
  static ptr_t theInstance;
};

#endif /* SINGLETON_HPP_ */
