/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LinkedCell_ControllerUnitTest.cpp
 *
 *  Created on: Nov 29, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Atom/atom.hpp"
#include "Box.hpp"
#include "CodePatterns/Assert.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinkedCell/LinkedCell_Controller.hpp"
#include "LinkedCell/LinkedCell_View.hpp"
#include "LinkedCell/LinkedCell_View_ModelWrapper.hpp"
#include "LinkedCell/unittests/defs.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"

#include "LinkedCell_ControllerUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( LinkedCell_ControllerTest );


void LinkedCell_ControllerTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create diag(20.) matrix
  BoxM = new RealSpaceMatrix;
  BoxM->setIdentity();
  (*BoxM) *= DOMAINLENGTH;

  // create Box with this matrix
  domain = new Box(*BoxM);

  controller = new LinkedCell::LinkedCell_Controller(*domain);

  // create empty set
  emptyset = new PointCloudAdaptor< std::vector<atom *> >(&emptyvector, std::string("emptyset"));
}


void LinkedCell_ControllerTest::tearDown()
{
  delete controller;
  delete domain;
  delete emptyset;
}

/** UnitTest for LinkedCell_Controller's lower and upper thresholds.
 */
void LinkedCell_ControllerTest::thresholdTest()
{
  /// re-create instances
  delete controller;
  delete domain;

  /// create diag(..) matrix beyond upper_threshold
  const double old_threshold = controller->upper_threshold;
  controller->lower_threshold = old_threshold*0.9;
  RealSpaceMatrix BoxM;
  BoxM.setIdentity();
  BoxM *= controller->upper_threshold*.5;

  /// create Box with this matrix
  domain = new Box(BoxM);

  controller = new LinkedCell::LinkedCell_Controller(*domain);

  /// check that thresholds have been adapted
  CPPUNIT_ASSERT( controller->upper_threshold != old_threshold );
  CPPUNIT_ASSERT( controller->lower_threshold != old_threshold*0.9 );
}

/** UnitTest for LinkedCell_Controller::getHeuristicRange().
 */
void LinkedCell_ControllerTest::getHeuristicRangeTest()
{
  /// re-implementing function to check is nonsense here, instead try some
  /// hard-coded, working values;
  controller->lower_threshold = 1.;
  controller->upper_threshold = 20.;
  const double inbetween = 9.5; // half and twice is definitely within both thresholds.

  /// check distance in between
  range<double> interval = controller->getHeuristicRange(inbetween);
  CPPUNIT_ASSERT ( interval.first != controller->lower_threshold );
  CPPUNIT_ASSERT ( interval.last != controller->upper_threshold );
}

/** UnitTest for LinkedCell_Controller::getViewTest() for getting twice the same view.
 */
void LinkedCell_ControllerTest::getView_SameViewTest()
{
  /// obtain a view
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(2., *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  {
    /// get same view again and check that now new instance appears
    LinkedCell::LinkedCell_View view_again = controller->getView(2., *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
}

/** UnitTest for LinkedCell_Controller::getViewTest() for picking two different views.
 */
void LinkedCell_ControllerTest::getView_DifferentViewTest()
{
  /// obtain a view
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(2., *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  {
    /// pick another view that is not close enough
    LinkedCell::LinkedCell_View view_other = controller->getView(5., *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, controller->ModelsMap.size() );
  }
}

/** UnitTest for LinkedCell_Controller::getViewTest() for picking further views in range of present one.
 */
void LinkedCell_ControllerTest::getView_InRangeViewTest()
{
  /// obtain a view
  const double edgelength = 2.;
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(edgelength, *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  /// pick views that are close enough
  range<double> interval = controller->getHeuristicRange(edgelength);
  {
    /// ... at half lower interval half
    LinkedCell::LinkedCell_View view_lowerhalf = controller->getView((edgelength + interval.first)/2., *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
  {
    /// ... at half upper interval half
    LinkedCell::LinkedCell_View view_upperhalf = controller->getView((interval.last + edgelength)/2., *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
  {
    /// ... close to lower boundary
    LinkedCell::LinkedCell_View view_closelower = controller->getView(interval.first + std::numeric_limits<double>::round_error(), *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
  {
    /// ... close to upper boundary
    LinkedCell::LinkedCell_View view_closerupper = controller->getView(interval.last - std::numeric_limits<double>::round_error(), *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
  {
    /// on lower boundary
    LinkedCell::LinkedCell_View view_onlower = controller->getView(interval.first, *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
}

/** UnitTest for LinkedCell_Controller::getViewTest() for picking further views outside range.
 */
void LinkedCell_ControllerTest::getView_OutOfRangeViewTest()
{
  /// Here we need half of the edge length to be greater than lower_threshold
  const double edgelength = 2.5;
  CPPUNIT_ASSERT( (edgelength/2.) > controller->lower_threshold );
  /// obtain a view
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(edgelength, *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  /// pick views that are not close enough and check for new instance
  range<double> interval = controller->getHeuristicRange(edgelength);
  {
    /// ... outside lower boundary
    LinkedCell::LinkedCell_View view_outsidelower = controller->getView(interval.first - std::numeric_limits<double>::round_error(), *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, controller->ModelsMap.size() );
  }
  {
    /// ... on upper boundary
    LinkedCell::LinkedCell_View view_onupper = controller->getView(interval.last, *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)3, controller->ModelsMap.size() );
  }
}

/** UnitTest for LinkedCell_Controller::getViewTest() for picking views beneath lower threshold.
 */
void LinkedCell_ControllerTest::getView_LowerThresholdViewTest()
{
  /// obtain a view
  const double edgelength = 1.9*controller->lower_threshold;
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(edgelength, *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  {
    /// get a view at threshold and check that no new instance has been created
    LinkedCell::LinkedCell_View view_onlower = controller->getView(controller->lower_threshold, *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
  {
    /// pick a view below 1.
    LinkedCell::LinkedCell_View view_beneathlower = controller->getView(0.1*controller->lower_threshold, *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }
}

/** UnitTest for LinkedCell_Controller::getViewTest() for picking views above upper threshold.
 */
void LinkedCell_ControllerTest::getView_UpperThresholdViewTest()
{
  /// obtain a view
  const double edgelength = controller->upper_threshold;
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(edgelength, *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );

  {
    /// get a view beyond threshold and check that no new instance has been created
    LinkedCell::LinkedCell_View view_beyondupper = controller->getView(1.1*controller->upper_threshold, *emptyset);
    CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  }

  {
    /// pick a view below threshold and check for new instance (if we make it outside acceptable range)
    range<double> interval = controller->getHeuristicRange(edgelength);
    if ( !interval.isInRange(0.1*controller->upper_threshold) ) {
      LinkedCell::LinkedCell_View view_beneathupper = controller->getView(0.1*controller->upper_threshold, *emptyset);
      CPPUNIT_ASSERT_EQUAL( (size_t)2, controller->ModelsMap.size() );
    }
  }
}

/** UnitTest for LinkedCell_Controller::updateModelsForNewBoxMatrix().
 */
void LinkedCell_ControllerTest::updateBoxTest()
{
  /// sign on to domain
  domain->signOn(controller, Box::MatrixChanged);

  /// obtain a view
  const double edgelength = controller->upper_threshold;
  CPPUNIT_ASSERT_EQUAL( (size_t)0, controller->ModelsMap.size() );
  LinkedCell::LinkedCell_View view = controller->getView(edgelength, *emptyset);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, controller->ModelsMap.size() );
  const LinkedCell::LinkedCell_Model * const model = view.LC->getModel();

  /// change box matrix
  domain->setM(*BoxM);

  /// check that model has changed
  CPPUNIT_ASSERT( model != view.LC->getModel() );

  /// sign off
  domain->signOff(controller, Box::MatrixChanged);
}
