/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomicInfo.cpp
 *
 *  Created on: Aug 10, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include "atom.hpp"
#include "AtomicInfo.hpp"
#include "CodePatterns/Assert.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Element/element.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"
#include "World.hpp"

/********************************** Functions for class AtomicInfo **********************************/

AtomicInfo::AtomicInfo() :
    Type(NULL),
    charge(0.),
    FatherId(0),
    MolId(0),
    Id(0),
    Nr(0)
{}

AtomicInfo::AtomicInfo(const atom &_atom) :
    Position(_atom.getPosition()),
    Type(_atom.getType()),
    charge(_atom.getCharge()),
    Velocity(_atom.getAtomicVelocity()),
    Force(_atom.getAtomicForce()),
    FatherId(_atom.father->getId()),
    MolId(0),
    Id(_atom.getId()),
    Nr(_atom.getNr())
{
  const molecule * const mol = _atom.getMolecule();
  ASSERT(mol != NULL,
      "Atom "+toString(_atom.getId())+" is not associated with any molecule.");
  MolId = mol->getId();
  // accumulate bond info
  const BondList &ListOfBonds = _atom.getListOfBonds();
  bonds.reserve(ListOfBonds.size());
  for (BondList::const_iterator bonditer = ListOfBonds.begin();
      bonditer != ListOfBonds.end(); ++bonditer) {
    const BondInfo bondinfo(*bonditer);
    bonds.push_back(bondinfo);
    LOG(3, "DEBUG: Storing info for bond " << bondinfo.leftid << "<->" << bondinfo.rightid);
  }
}

AtomicInfo::AtomicInfo(const AtomicInfo &_atominfo) :
    Position(_atominfo.Position),
    Type(_atominfo.Type),
    charge(_atominfo.charge),
    Velocity(_atominfo.Velocity),
    Force(_atominfo.Force),
    FatherId(_atominfo.FatherId),
    MolId(_atominfo.MolId),
    Id(_atominfo.Id),
    Nr(_atominfo.Nr),
    bonds(_atominfo.bonds.begin(), _atominfo.bonds.end())
{}

AtomicInfo::~AtomicInfo()
{}


bool AtomicInfo::setAtom(atom &_atom) const
{
  _atom.setPosition(Position);
  _atom.setType(Type);
  _atom.setCharge(charge);
  _atom.setAtomicVelocity(Velocity);
  _atom.setAtomicForce(Force);

  // set old id
  bool status = true;
  if (_atom.getId() != Id)
    status &= _atom.changeId(Id);

  // set its father
  if (status) {
    atom * const _father = World::getInstance().getAtom(AtomById(FatherId));
    if ((_father == NULL) || (Id == FatherId)) {
      _atom.father = &_atom;
      // don't sign on
    } else {
      _atom.father = _father;
      _father->signOn(&_atom);
    }

    // set bonds
    for (std::vector<BondInfo>::const_iterator bonditer = bonds.begin();
        bonditer != bonds.end(); ++bonditer)
      (*bonditer).RecreateBond();

    // setting molecule
    molecule * const _mol = World::getInstance().getMolecule(MoleculeById(MolId));
    if (_mol != NULL)
      _atom.setMolecule(_mol); // this is ok, mol is const within AtomicInfo, but not outside (atoms need to register)
    _atom.changeNr(Nr);
  }

  return status;
}

atomId_t AtomicInfo::getId() const {
  return Id;
}

AtomicInfo& AtomicInfo::operator=(const AtomicInfo& AI)
{
  if (&AI == this)  // check self-assignment
    return *this;
  Position = AI.Position;
  Type = AI.Type;
  FatherId = AI.FatherId;
  MolId = AI.MolId;
  Velocity = AI.Velocity;
  Force = AI.Force;
  Id = AI.Id;
  Nr = AI.Nr;
  return *this;
}

