/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * NotMoleculeByOrderAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "molecule.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeOrderDescriptor.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "NotMoleculeByOrderAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "NotMoleculeByOrderAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

ActionState::ptr SelectionNotMoleculeByOrderAction::performCall() {
  size_t no_unselected = 0;
  const std::vector<int> &indices = params.molindices.get();
  std::vector<const molecule *> mols;
  for( std::vector<int>::const_iterator iter = indices.begin();
      iter != indices.end(); ++iter) {
    const molecule *mol = const_cast<const World &>(World::getInstance()).
        getMolecule(MoleculeByOrder(*iter));

    if (mol != NULL) {
      if (World::getInstance().isSelected(mol)) {
        //LOG(1, "Unselecting molecule " << mol->name);
        World::getInstance().unselectMolecule(mol);
        mols.push_back(mol);
        ++no_unselected;
      }
    } else {
      STATUS("Cannot find molecule by given index "+toString(*iter)+".");
      return Action::failure;
    }
  }

  LOG(0, no_unselected << " molecules additionally unselected.");
  if (no_unselected != 0) {
    return ActionState::ptr(new SelectionNotMoleculeByOrderState(mols, params));
  } else {
    return Action::success;
  }
}

ActionState::ptr SelectionNotMoleculeByOrderAction::performUndo(ActionState::ptr _state) {
  SelectionNotMoleculeByOrderState *state = assert_cast<SelectionNotMoleculeByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<const molecule *>::const_iterator iter = state->mols.begin();
      iter != state->mols.end(); ++iter)
    world.selectMolecule(*iter);

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionNotMoleculeByOrderAction::performRedo(ActionState::ptr _state){
  SelectionNotMoleculeByOrderState *state = assert_cast<SelectionNotMoleculeByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<const molecule *>::const_iterator iter = state->mols.begin();
      iter != state->mols.end(); ++iter)
    world.unselectMolecule(*iter);

  return ActionState::ptr(_state);
}

bool SelectionNotMoleculeByOrderAction::canUndo() {
  return true;
}

bool SelectionNotMoleculeByOrderAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
