/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MoleculeByOrderAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "molecule.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeOrderDescriptor.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "MoleculeByOrderAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "MoleculeByOrderAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

ActionState::ptr SelectionMoleculeByOrderAction::performCall() {
  size_t no_selected = 0;
  const std::vector<int> &indices = params.molindices.get();
  std::vector<const molecule *> mols;
  const World &const_world = World::getConstInstance();
  World &world = World::getInstance();
  for( std::vector<int>::const_iterator iter = indices.begin(); 
      iter != indices.end(); ++iter) {
    const molecule *mol = const_world.getMolecule(MoleculeByOrder(*iter));

    if (mol != NULL) {
      if (!const_world.isSelected(mol)) {
        //LOG(1, "Selecting molecule " << mol->name);
        world.selectMolecule(mol);
        mols.push_back(mol);
        ++no_selected;
      }
    } else {
      STATUS("Cannot find molecule by given index "+toString(*iter)+".");
      return Action::failure;
    }
  }

  LOG(0, no_selected << " molecules additionally selected.");
  if (no_selected != 0) {
    return ActionState::ptr(new SelectionMoleculeByOrderState(mols, params));
  } else {
    return Action::success;
  }
}

ActionState::ptr SelectionMoleculeByOrderAction::performUndo(ActionState::ptr _state) {
  SelectionMoleculeByOrderState *state = assert_cast<SelectionMoleculeByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<const molecule *>::const_iterator iter = state->mols.begin();
      iter != state->mols.end(); ++iter)
    world.unselectMolecule(*iter);

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionMoleculeByOrderAction::performRedo(ActionState::ptr _state){
  SelectionMoleculeByOrderState *state = assert_cast<SelectionMoleculeByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<const molecule *>::const_iterator iter = state->mols.begin();
      iter != state->mols.end(); ++iter)
    world.selectMolecule(*iter);
  return ActionState::ptr(_state);
}

bool SelectionMoleculeByOrderAction::canUndo() {
  return true;
}

bool SelectionMoleculeByOrderAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
