/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomByOrderAction.cpp
 *
 *  Created on: Mar 22, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "World.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/AtomOrderDescriptor.hpp"

#include <iostream>
#include <string>

#include "AtomByOrderAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AtomByOrderAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr SelectionAtomByOrderAction::performCall() {
  size_t no_selected = 0;
  const std::vector<int> &indices = params.orders.get();
  std::vector<atomId_t> atomids;
  const World &const_world = World::getConstInstance();
  World &world = World::getInstance();
  for( std::vector<int>::const_iterator iter = indices.begin();
      iter != indices.end(); ++iter) {
    const atom * const walker = const_world.getAtom(AtomByOrder(*iter));

    if (walker != NULL) {
      if (!const_world.isSelected(walker)) {
        //LOG(1, "Selecting atom " << *walker);
        world.selectAtom(walker);
        atomids.push_back(walker->getId());
        ++no_selected;
      }
    } else {
      STATUS("Cannot find atom by given index "+toString(*iter)+".");
      return Action::failure;
    }
  }

  LOG(0, no_selected << " atoms additionally selected.");
  if (no_selected != 0) {
    return ActionState::ptr(new SelectionAtomByOrderState(atomids, params));
  } else {
    return Action::success;
  }
}

ActionState::ptr SelectionAtomByOrderAction::performUndo(ActionState::ptr _state) {
  SelectionAtomByOrderState *state = assert_cast<SelectionAtomByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<atomId_t>::const_iterator iter = state->WalkerIds.begin();
      iter != state->WalkerIds.end(); ++iter)
    world.unselectAllAtoms(AtomById(*iter));

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionAtomByOrderAction::performRedo(ActionState::ptr _state){
  SelectionAtomByOrderState *state = assert_cast<SelectionAtomByOrderState*>(_state.get());

  World &world = World::getInstance();
  for (std::vector<atomId_t>::const_iterator iter = state->WalkerIds.begin();
      iter != state->WalkerIds.end(); ++iter)
    world.selectAllAtoms(AtomById(*iter));

  return ActionState::ptr(_state);
}

bool SelectionAtomByOrderAction::canUndo() {
  return true;
}

bool SelectionAtomByOrderAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
