/*
 * Cacheable.hpp
 *
 *  Created on: Feb 2, 2010
 *      Author: crueger
 */

#ifndef CACHEABLE_HPP_
#define CACHEABLE_HPP_

#include "Patterns/Observer.hpp"
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include <iostream>

#include "Helpers/Assert.hpp"

#ifndef NO_CACHING

  template<typename T>
  class Cacheable : public Observer
  {
    // we define the states of the cacheable so we can do very fast state-checks
    class State{
    public:
      State(Cacheable *_owner) :
        owner(_owner),
        busy(false)
        {}
      virtual T& getValue()=0;
      virtual void invalidate()=0;
      virtual bool isValid()=0;
      virtual void enter()=0;
      bool isBusy(){
        return busy;
      }
    protected:
      bool busy;
      Cacheable *owner;
    };

    class InvalidState : public State{
    public:
      InvalidState(Cacheable *_owner):
        State(_owner)
        {}

      virtual T& getValue(){
        // set the state to valid
        State::owner->switchState(State::owner->validState);
        // get the value from the now valid state
        return State::owner->state->getValue();
      }

      virtual void invalidate(){
        // nothing to do on this message
      }

      virtual bool isValid(){
        return false;
      }

      virtual void enter(){
        // nothing to do when entering this
      }
    };

    class ValidState : public State{
    public:
      ValidState(Cacheable *_owner) :
        State(_owner)
        {}

      virtual T& getValue(){
        return content;
      }

      virtual void invalidate(){
        State::owner->switchState(State::owner->invalidState);
      }

      virtual bool isValid(){
        return true;
      }

      virtual void enter(){
        State::busy= true;
        // as soon as we enter the valid state we recalculate
        content = State::owner->recalcMethod();
        State::busy = false;
      }
    private:
      T content;
    };

    class DestroyedState : public State {
    public:
      DestroyedState(Cacheable *_owner) :
        State(_owner)
        {}

      virtual T& getValue(){
        ASSERT(0,"Cannot get a value from a Cacheable after it's Observable has died");
        // we have to return a grossly invalid reference, because no value can be produced anymore
        return *(static_cast<T*>(0));
      }

      virtual void invalidate(){
        ASSERT(0,"Cannot invalidate a Cacheable after it's Observable has died");
      }

      virtual bool isValid(){
        ASSERT(0,"Cannot check validity of a Cacheable after it's Observable has died");
        return false;
      }

      virtual void enter(){
        // nothing to do when entering this state
      }
    };


  typedef boost::shared_ptr<State> state_ptr;

  public:
    Cacheable(Observable *_owner, boost::function<T()> _recalcMethod);
    virtual ~Cacheable();

    const bool isValid() const;
    const T& operator*() const;

    // methods implemented for base-class Observer
    void update(Observable *subject);
    void subjectKilled(Observable *subject);
  private:

    void switchState(state_ptr newState);

    mutable state_ptr state;
    // pre-defined state so we don't have to construct to much
    state_ptr invalidState;
    state_ptr validState;
    // destroyed state is not predefined, because we rarely enter that state and never leave

    Observable *owner;

    boost::function<T()> recalcMethod;

    // de-activated copy constructor
    Cacheable(const Cacheable&);
  };


  template<typename T>
  Cacheable<T>::Cacheable(Observable *_owner, boost::function<T()> _recalcMethod) :
    owner(_owner),
    recalcMethod(_recalcMethod)
  {
    // create all states needed for this object
    invalidState = state_ptr(new InvalidState(this));
    validState = state_ptr(new ValidState(this));
    state = invalidState;
    // we sign on with the best(=lowest) priority, so cached values are recalculated before
    // anybody else might ask for updated values
    owner->signOn(this,-20);
  }

  // de-activated copy constructor
  template<typename T>
  Cacheable<T>::Cacheable(const Cacheable&){
    ASSERT(0,"Cacheables should never be copied");
  }

  template<typename T>
  const T& Cacheable<T>::operator*() const{
    return state->getValue();
  }

  template<typename T>
  Cacheable<T>::~Cacheable()
  {
    owner->signOff(this);
  }

  template<typename T>
  const bool Cacheable<T>::isValid() const{
    return state->isValid();
  }

  template<typename T>
  void Cacheable<T>::update(Observable *subject) {
    state->invalidate();
  }

  template<typename T>
  void Cacheable<T>::subjectKilled(Observable *subject) {
    state_ptr destroyed = state_ptr(new DestroyedState(this));
    switchState(destroyed);
  }

  template<typename T>
  void Cacheable<T>::switchState(state_ptr newState){
    ASSERT(!state->isBusy(),"LOOP DETECTED: Cacheable state switched while recalculating.\nDid the recalculation trigger the Observable?");
    state = newState;
    state->enter();
  }

#else
  template<typename T>
  class Cacheable : public Observer
  {
  public:
    Cacheable(Observable *_owner, boost::function<T()> _recalcMethod);
    virtual ~Cacheable();

    const bool isValid() const;
    const T& operator*() const;

    // methods implemented for base-class Observer
    void update(Observable *subject);
    void subjectKilled(Observable *subject);
  private:

    boost::function<T()> recalcMethod;
  };

  template<typename T>
  Cacheable<T>::Cacheable(Observable *_owner, boost::function<T()> _recalcMethod) :
    recalcMethod(_recalcMethod)
  {}

  template<typename T>
  const T& Cacheable<T>::operator*() const{
    return recalcMethod();
  }

  template<typename T>
  Cacheable<T>::~Cacheable()
  {}

  template<typename T>
  const bool Cacheable<T>::isValid() const{
    return true;
  }

  template<typename T>
  void Cacheable<T>::update(Observable *subject) {
    ASSERT(0, "Cacheable::update should never be called when caching is disabled");
  }

  template<typename T>
  void Cacheable<T>::subjectKilled(Observable *subject){
    ASSERT(0, "Cacheable::subjectKilled should never be called when caching is disabled");
  }
#endif

#endif /* CACHEABLE_HPP_ */
