/*
 * LinkedCell_Model.hpp
 *
 *  Created on: Nov 15, 2011
 *      Author: heber
 */

#ifndef LINKEDCELL_MODEL_HPP_
#define LINKEDCELL_MODEL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/multi_array.hpp>
#include <map>

#include "CodePatterns/Observer/Observer.hpp"
#include "LinearAlgebra/defs.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinkedCell/types.hpp"

class Box;
class IPointCloud;
class LinkedCell_ModelTest;
class TesselPoint;
class Vector;

namespace LinkedCell {

  /** This is the model in the MVC ansatz for the LinkedCell structure.
   *
   * \sa linkedcell
   *
   * The model represents a certain linked cell structure with a specific mesh
   * size (i.e. edge length). Note that all coordinates are internally always
   * transformed to [0,1]^3, hence we require the domain (\ref Box) for the
   * transformation matrices.
   *
   * This class stores internally list of particles in three-dimensional arrays.
   *
   */
  class LinkedCell_Model : public Observer
  {
    //!> grant unit test access to internal structure
    friend class ::LinkedCell_ModelTest;
  public:
    LinkedCell_Model(const double radius, const Box &_domain);
    LinkedCell_Model(IPointCloud &set, const double radius, const Box &_domain);
    ~LinkedCell_Model();

    typedef std::pair<tripleIndex,tripleIndex> LinkedCellNeighborhoodBounds;

    const tripleIndex getIndexToVector(const Vector &position) const;
    const LinkedCellNeighborhoodBounds getNeighborhoodBounds(const tripleIndex &index, const tripleIndex &step = NearestNeighbors) const;
    const tripleIndex getStep(const double distance) const;
    const LinkedCell& getCell(const tripleIndex &index) const;
    bool checkArrayBounds(const tripleIndex &index) const;
    void applyBoundaryConditions(tripleIndex &index) const;

    void addNode(const TesselPoint *Walker);
    void deleteNode(const TesselPoint *Walker);
    void moveNode(const TesselPoint *Walker);

    void setPartition(double distance);

    //!> static indices to get nearest neighbors as default argument
    static tripleIndex NearestNeighbors;

  protected:
    void update(Observable *publisher);
    void recieveNotification(Observable *publisher, Notification_ptr notification);
    void subjectKilled(Observable *publisher);

  private:
    void AllocateCells();
    void Reset();
    void insertPointCloud(IPointCloud &set);

    void startListening();
    void stopListening();

    LinkedCellArray::index getSize(const size_t dim) const;

    typedef LinkedCellArray::size_type size_type;
    typedef LinkedCellArray::iterator iterator3;
    typedef boost::subarray_gen<LinkedCellArray, NDIM-1>::type::iterator iterator2;
    typedef boost::subarray_gen<LinkedCellArray, NDIM-2>::type::iterator iterator1;

    typedef std::map<const TesselPoint *, LinkedCell *> MapPointToCell;

    //!> internal shape of multi_array
    size_type *internal_Sizes;

    //!> internal index of current cell, this makes going through linked cell faster
    tripleIndex internal_index;

    //!> Lookup map to get the cell for a given TesselPoint
    MapPointToCell CellLookup;

    //!> edge length per axis
    boost::array<double, 3> EdgeLength;

    //!> Linked cell array
    LinkedCellArray N;

    //!> matrix to divide Box with
    RealSpaceMatrix Dimensions;

    //!> matrix to transform normal position to obtain partitioned position
    RealSpaceMatrix Partition;

    //!> Box with matrix transformations and boundary conditions
    const Box &domain;
  };

}

// inlined functions
#include "LinkedCell_Model_inline.hpp"

#endif /* LINKEDCELL_MODEL_HPP_ */
