/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RandomNumberEngineFactoryUnitTest.cpp
 *
 *  Created on: Jan 03, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Assert.hpp"

#include "RandomNumberEngineFactoryUnitTest.hpp"

#include "RandomNumbers/RandomNumberEngine.hpp"
#include "RandomNumbers/RandomNumberEngine_Encapsulation.hpp"
#include "RandomNumbers/RandomNumberEngineFactory.hpp"

#include <boost/nondet_random.hpp>
#include <boost/random.hpp>
#include <boost/random/additive_combine.hpp>
#include <boost/random/discard_block.hpp>
#include <boost/random/inversive_congruential.hpp>
#include <boost/random/lagged_fibonacci.hpp>
#include <boost/random/linear_congruential.hpp>
#include <boost/random/linear_feedback_shift.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/random_number_generator.hpp>
#include <boost/random/ranlux.hpp>
#include <boost/random/shuffle_output.hpp>
#include <boost/random/subtract_with_carry.hpp>
#include <boost/random/xor_combine.hpp>
#include <boost/random/bernoulli_distribution.hpp>
#include <boost/random/binomial_distribution.hpp>
#include <boost/random/cauchy_distribution.hpp>
#include <boost/random/exponential_distribution.hpp>
#include <boost/random/gamma_distribution.hpp>
#include <boost/random/geometric_distribution.hpp>
#include <boost/random/linear_congruential.hpp>
#include <boost/random/lognormal_distribution.hpp>
#include <boost/random/normal_distribution.hpp>
#include <boost/random/poisson_distribution.hpp>
#include <boost/random/triangle_distribution.hpp>
#include <boost/random/uniform_01.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/random/uniform_on_sphere.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/uniform_smallint.hpp>

#include <typeinfo>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( RandomNumberEngineFactoryTest );

void RandomNumberEngineFactoryTest::setUp()
{
  rndA = NULL;
  rndA_1 = NULL;
  rndA_2 = NULL;
  rndA_3 = NULL;
  RandomNumberEngineFactory::getInstance();
}

void RandomNumberEngineFactoryTest::tearDown()
{
  delete rndA;
  delete rndA_1;
  delete rndA_2;
  delete rndA_3;
  RandomNumberEngineFactory::purgeInstance();
}

void RandomNumberEngineFactoryTest::EngineTest()
{
  // check the injectiveness of enum and string map
  for (RandomNumberEngineFactory::NameMap::const_iterator
      iter = RandomNumberEngineFactory::getInstance().names.begin();
      iter != RandomNumberEngineFactory::getInstance().names.end();
      ++iter) {
    CPPUNIT_ASSERT_EQUAL(
        iter->second,
        RandomNumberEngineFactory::getInstance().getName(
            RandomNumberEngineFactory::getInstance().getEnum(
                iter->second
            )
        )
    );
  }

  // check one of the engines in the table
  rndA = RandomNumberEngineFactory::getInstance().
      ManipulablePrototypeTable[RandomNumberEngineFactory::minstd_rand0]->clone();
  CPPUNIT_ASSERT_EQUAL(
      std::string(typeid(boost::minstd_rand0).name()),
      rndA->name()
  );

  // range of minstd_rand0 with c=0 is [1,m)
  CPPUNIT_ASSERT_EQUAL( 1., rndA->min() );
}

void RandomNumberEngineFactoryTest::PrototypeManipulationTest()
{
  // make unmodified clone
  rndA_1 = RandomNumberEngineFactory::getInstance().
      getProduct(RandomNumberEngineFactory::minstd_rand0);

  // obtain manipulator
  RandomNumberEngine &prototype = RandomNumberEngineFactory::getInstance().
      getPrototype(RandomNumberEngineFactory::minstd_rand0);

  // change the prototype directly
  CPPUNIT_ASSERT ( 0 != prototype.getseed() );
  prototype.seed(0);

  // check that prototype has indeed been manipulated
  rndA_2 = RandomNumberEngineFactory::getInstance().
      getProduct(RandomNumberEngineFactory::minstd_rand0);
  CPPUNIT_ASSERT_EQUAL( (unsigned int)0, rndA_2->getseed() );
  CPPUNIT_ASSERT( rndA_2->getseed() != rndA_1->getseed());

  // manipulate prototype
  RandomNumberEngine_Parameters *params = rndA_1->getParameterSet();
  CPPUNIT_ASSERT ( rndA_1->getseed() != (unsigned int)3 );
  params->seed = 3;
  RandomNumberEngineFactory::getInstance().
        manipulatePrototype(RandomNumberEngineFactory::minstd_rand0, *params);

  //
  rndA_3 = RandomNumberEngineFactory::getInstance().
      getProduct(RandomNumberEngineFactory::minstd_rand0);
  CPPUNIT_ASSERT_EQUAL( (unsigned int)3, rndA_3->getseed() );
  CPPUNIT_ASSERT( rndA_3->getseed() != rndA_1->getseed());

  delete params;
}

